// Forerunner Protocol Background Service Worker

let activeConnections = {};
let metrics = {
  sessions: 0,
  messagesSent: 0,
  messagesReceived: 0,
  bytesSent: 0,
  bytesReceived: 0
};

// Handle extension installation
chrome.runtime.onInstalled.addListener(() => {
  console.log('Forerunner Protocol extension installed');
  chrome.storage.local.set({ metrics });
});

// Handle messages from content scripts
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.type) {
    case 'CONNECT':
      handleConnect(message.data, sendResponse);
      return true;

    case 'SEND_MESSAGE':
      handleSendMessage(message.data, sendResponse);
      return true;

    case 'GET_METRICS':
      sendResponse({ success: true, data: metrics });
      return false;

    case 'DISCONNECT':
      handleDisconnect(message.data, sendResponse);
      return true;

    default:
      sendResponse({ success: false, error: 'Unknown message type' });
      return false;
  }
});

async function handleConnect(data, sendResponse) {
  const { sessionId, validatorUrl } = data;

  try {
    const ws = new WebSocket(validatorUrl);

    ws.onopen = () => {
      activeConnections[sessionId] = ws;
      metrics.sessions++;
      saveMetrics();
      sendResponse({ success: true, sessionId });
    };

    ws.onmessage = (event) => {
      metrics.messagesReceived++;
      metrics.bytesReceived += event.data.length;
      saveMetrics();

      // Forward message to content script
      chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (tabs[0]) {
          chrome.tabs.sendMessage(tabs[0].id, {
            type: 'FORERUNNER_MESSAGE',
            sessionId,
            data: event.data
          });
        }
      });
    };

    ws.onerror = (error) => {
      console.error('WebSocket error:', error);
      sendResponse({ success: false, error: error.message });
    };

    ws.onclose = () => {
      delete activeConnections[sessionId];
    };

  } catch (error) {
    sendResponse({ success: false, error: error.message });
  }
}

function handleSendMessage(data, sendResponse) {
  const { sessionId, message } = data;
  const ws = activeConnections[sessionId];

  if (!ws || ws.readyState !== WebSocket.OPEN) {
    sendResponse({ success: false, error: 'Not connected' });
    return;
  }

  try {
    ws.send(message);
    metrics.messagesSent++;
    metrics.bytesSent += message.length;
    saveMetrics();
    sendResponse({ success: true });
  } catch (error) {
    sendResponse({ success: false, error: error.message });
  }
}

function handleDisconnect(data, sendResponse) {
  const { sessionId } = data;
  const ws = activeConnections[sessionId];

  if (ws) {
    ws.close();
    delete activeConnections[sessionId];
    sendResponse({ success: true });
  } else {
    sendResponse({ success: false, error: 'Session not found' });
  }
}

function saveMetrics() {
  chrome.storage.local.set({ metrics });
}

// Periodic cleanup of stale connections
setInterval(() => {
  Object.keys(activeConnections).forEach(sessionId => {
    const ws = activeConnections[sessionId];
    if (ws.readyState === WebSocket.CLOSED || ws.readyState === WebSocket.CLOSING) {
      delete activeConnections[sessionId];
    }
  });
}, 30000);

console.log('Forerunner Protocol background service worker loaded');
